<?php
// Include configuration file
require_once 'config.php';

// Require user to be logged in
require_login();

// Set the current page for navigation
$current_page = 'settings';

// Initialize connection
$conn = connect_db();

// Initialize variables
$settings = [
    'app_name' => $config['app_name'],
    'app_version' => $config['app_version'] ?? '1.0.0',
    'default_pagination' => '10',
    'footer_text' => 'Namibia Statistics Agency',
    'enable_logs' => '1',
    'logo_path' => 'assets/nsa-logo.jpeg'
];

$error = '';
$success = '';

// Load settings from the database if available
$result = $conn->query("SHOW TABLES LIKE 'system_settings'");
if ($result->num_rows > 0) {
    $result = $conn->query("SELECT * FROM system_settings");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Process form data
    $new_settings = [
        'app_name' => sanitize_input($_POST['app_name']),
        'footer_text' => sanitize_input($_POST['footer_text']),
        'default_pagination' => (int)$_POST['default_pagination'],
        'enable_logs' => isset($_POST['enable_logs']) ? '1' : '0'
    ];
    
    // Handle logo upload if provided
    if (isset($_FILES['logo']) && $_FILES['logo']['error'] == 0) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['logo']['name'];
        $ext = pathinfo($filename, PATHINFO_EXTENSION);
        
        if (in_array(strtolower($ext), $allowed)) {
            // Create assets directory if it doesn't exist
            if (!file_exists('assets')) {
                mkdir('assets', 0755, true);
            }
            
            $new_filename = 'nsa-logo.' . $ext;
            $upload_path = 'assets/' . $new_filename;
            
            if (move_uploaded_file($_FILES['logo']['tmp_name'], $upload_path)) {
                $new_settings['logo_path'] = $upload_path;
            } else {
                $error = "Failed to upload logo.";
            }
        } else {
            $error = "Invalid logo file type. Allowed: jpg, jpeg, png, gif";
        }
    } else {
        $new_settings['logo_path'] = $settings['logo_path'];
    }
    
    // Save settings to the database
    if (empty($error)) {
        // Check if settings table exists, if not create it
        $result = $conn->query("SHOW TABLES LIKE 'system_settings'");
        if ($result->num_rows == 0) {
            $conn->query("CREATE TABLE system_settings (
                id INT AUTO_INCREMENT PRIMARY KEY,
                setting_key VARCHAR(50) NOT NULL UNIQUE,
                setting_value TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )");
        }
        
        // Update or insert settings
        foreach ($new_settings as $key => $value) {
            $stmt = $conn->prepare("INSERT INTO system_settings (setting_key, setting_value) 
                                    VALUES (?, ?) 
                                    ON DUPLICATE KEY UPDATE setting_value = ?");
            $stmt->bind_param("sss", $key, $value, $value);
            $stmt->execute();
            $stmt->close();
        }
        
        // Update local settings array
        $settings = array_merge($settings, $new_settings);
        
        // Also update config.php for app_name
        $config_file = file_get_contents('config.php');
        $config_file = preg_replace(
            "/'app_name'\s*=>\s*'([^']*)'/", 
            "'app_name' => '" . addslashes($new_settings['app_name']) . "'", 
            $config_file
        );
        file_put_contents('config.php', $config_file);
        
        $success = "Settings updated successfully.";
        log_action("Updated system settings");
    }
}

// Get database size
$db_size = 0;
$result = $conn->query("SELECT SUM(data_length + index_length) as size FROM information_schema.TABLES WHERE table_schema = '" . $config['db_name'] . "'");
if ($result && $row = $result->fetch_assoc()) {
    $db_size = $row['size'];
}

// Get PHP and MySQL versions
$php_version = phpversion();
$mysql_version = $conn->server_info;

// Count the number of reports, indicators, categories, themes, and users
$stats = [];

$result = $conn->query("SELECT COUNT(*) as count FROM reports");
if ($result && $row = $result->fetch_assoc()) {
    $stats['reports'] = $row['count'];
}

$result = $conn->query("SELECT COUNT(*) as count FROM national_indicators");
if ($result && $row = $result->fetch_assoc()) {
    $stats['indicators'] = $row['count'];
}

$result = $conn->query("SELECT COUNT(*) as count FROM report_categories");
if ($result && $row = $result->fetch_assoc()) {
    $stats['categories'] = $row['count'];
}

$result = $conn->query("SELECT COUNT(*) as count FROM indicator_themes");
if ($result && $row = $result->fetch_assoc()) {
    $stats['themes'] = $row['count'];
}

$result = $conn->query("SELECT COUNT(*) as count FROM users");
if ($result && $row = $result->fetch_assoc()) {
    $stats['users'] = $row['count'];
}

// Close connection
$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - <?php echo $config['app_name']; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <style>
        :root {
            --primary-color: #1f3889;
            --secondary-color: #ad9237;
            --primary-dark: #152a6a;
            --secondary-dark: #967d2f;
            --sidebar-width: 250px;
        }
        
        body {
            min-height: 100vh;
            background-color: #f8f9fa;
        }
        
        /* Sidebar */
        .sidebar {
            width: var(--sidebar-width);
            position: fixed;
            top: 0;
            left: 0;
            height: 100%;
            z-index: 1000;
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            padding-top: 20px;
            transition: all 0.3s;
        }
        
        .sidebar-header {
            padding: 10px 20px;
            text-align: center;
            margin-bottom: 20px;
        }
        
        .sidebar-header img {
            max-width: 120px;
            margin-bottom: 10px;
        }
        
        .sidebar-menu {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .sidebar-menu li {
            margin-bottom: 5px;
        }
        
        .sidebar-menu a {
            color: rgba(255, 255, 255, 0.8);
            text-decoration: none;
            padding: 12px 20px;
            display: block;
            transition: all 0.3s;
            border-left: 4px solid transparent;
        }
        
        .sidebar-menu a:hover, .sidebar-menu a.active {
            color: white;
            background-color: rgba(255, 255, 255, 0.1);
            border-left-color: var(--secondary-color);
        }
        
        .sidebar-menu i {
            margin-right: 10px;
            width: 20px;
            text-align: center;
        }
        
        /* Main content */
        .main-content {
            margin-left: var(--sidebar-width);
            padding: 20px;
            transition: all 0.3s;
        }
        
        /* Header */
        .main-header {
            background-color: white;
            padding: 15px 20px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
            border-radius: 5px;
        }
        
        .main-header h1 {
            color: var(--primary-color);
            font-size: 1.5rem;
            margin-bottom: 0;
        }
        
        /* Cards */
        .content-card {
            background-color: white;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
            margin-bottom: 20px;
        }
        
        .content-card .card-header {
            padding: 15px 20px;
            border-bottom: 1px solid #eee;
        }
        
        .content-card .card-body {
            padding: 20px;
        }
        
        /* Tables */
        .data-table th {
            background-color: #f8f9fa;
        }
        
        /* Buttons */
        .btn-primary {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }
        
        .btn-primary:hover, .btn-primary:focus {
            background-color: var(--primary-dark);
            border-color: var(--primary-dark);
        }
        
        .btn-secondary {
            background-color: var(--secondary-color);
            border-color: var(--secondary-color);
        }
        
        .btn-secondary:hover, .btn-secondary:focus {
            background-color: var(--secondary-dark);
            border-color: var(--secondary-dark);
        }
        
        /* Footer */
        .footer {
            margin-left: var(--sidebar-width);
            padding: 15px 20px;
            text-align: center;
            color: #6c757d;
            font-size: 0.9rem;
            border-top: 1px solid #eee;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .sidebar {
                width: 0;
                padding-top: 60px;
            }
            
            .sidebar.active {
                width: var(--sidebar-width);
            }
            
            .main-content, .footer {
                margin-left: 0;
            }
            
            .main-content.sidebar-active, .footer.sidebar-active {
                margin-left: var(--sidebar-width);
            }
            
            .sidebar-toggle {
                display: block;
            }
        }
        
        .sidebar-toggle {
            display: none;
            position: fixed;
            top: 15px;
            left: 15px;
            z-index: 1050;
            background-color: var(--primary-color);
            border: none;
            color: white;
            width: 40px;
            height: 40px;
            border-radius: 3px;
        }
        
        /* Settings styles */
        .nav-pills .nav-link.active, .nav-pills .show>.nav-link {
            background-color: var(--primary-color);
        }
        
        .nav-pills .nav-link {
            color: var(--primary-color);
        }
        
        .logo-preview {
            max-width: 120px;
            max-height: 60px;
            margin-top: 10px;
            border: 1px solid #ddd;
            padding: 5px;
            border-radius: 4px;
        }
        
        .stat-item {
            padding: 15px;
            border-bottom: 1px solid #eee;
        }
        
        .stat-item:last-child {
            border-bottom: none;
        }
        
        .stat-label {
            color: #6c757d;
            font-size: 0.9rem;
        }
        
        .stat-value {
            font-size: 1.1rem;
            font-weight: 500;
        }
        
        .system-info {
            background-color: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 15px;
        }
    </style>
</head>
<body>
    <!-- Sidebar toggle button (mobile) -->
    <button class="sidebar-toggle d-md-none" id="sidebarToggle">
        <i class="fas fa-bars"></i>
    </button>
    
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="<?php echo htmlspecialchars($settings['logo_path'] ?? 'assets/nsa-logo.jpeg'); ?>" alt="NSA Logo" class="img-fluid" onerror="this.src='assets/nsa-logo.jpeg'">
            <h5>NSA Admin</h5>
        </div>
        
        <ul class="sidebar-menu">
            <li><a href="index.php"><i class="fas fa-tachometer-alt"></i> Dashboard</a></li>
            <li><a href="reports.php"><i class="fas fa-file-alt"></i> Reports</a></li>
            <li><a href="report-categories.php"><i class="fas fa-tags"></i> Report Categories</a></li>
            <li><a href="indicators.php"><i class="fas fa-chart-line"></i> Indicators</a></li>
            <li><a href="indicator-themes.php"><i class="fas fa-palette"></i> Indicator Themes</a></li>
            <li><a href="users.php"><i class="fas fa-users"></i> Users</a></li>
            <li><a href="settings.php" class="active"><i class="fas fa-cog"></i> Settings</a></li>
            <li><a href="login.php?action=logout"><i class="fas fa-sign-out-alt"></i> Logout</a></li>
        </ul>
    </div>
    
    <!-- Main content -->
    <div class="main-content">
        <!-- Header -->
        <div class="main-header d-flex justify-content-between align-items-center">
            <h1>System Settings</h1>
            <div class="user-info">
                <span class="me-2"><?php echo $_SESSION['username'] ?? 'Admin User'; ?></span>
                <i class="fas fa-user-circle"></i>
            </div>
        </div>
        
        <?php if (!empty($success)): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $success; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($error)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $error; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        <?php endif; ?>
        
        <div class="content-card">
            <div class="card-body">
                <ul class="nav nav-pills mb-4" id="settings-tab" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="general-tab" data-bs-toggle="pill" data-bs-target="#general" type="button" role="tab" aria-controls="general" aria-selected="true">
                            <i class="fas fa-cog me-1"></i> General
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="system-info-tab" data-bs-toggle="pill" data-bs-target="#system-info" type="button" role="tab" aria-controls="system-info" aria-selected="false">
                            <i class="fas fa-info-circle me-1"></i> System Info
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="database-tab" data-bs-toggle="pill" data-bs-target="#database" type="button" role="tab" aria-controls="database" aria-selected="false">
                            <i class="fas fa-database me-1"></i> Database
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="backup-tab" data-bs-toggle="pill" data-bs-target="#backup" type="button" role="tab" aria-controls="backup" aria-selected="false">
                            <i class="fas fa-download me-1"></i> Backup & Restore
                        </button>
                    </li>
                </ul>
                
                <div class="tab-content" id="settings-tabContent">
                    <!-- General Settings -->
                    <div class="tab-pane fade show active" id="general" role="tabpanel" aria-labelledby="general-tab">
                        <form action="settings.php" method="post" enctype="multipart/form-data">
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="app_name" class="form-label">Application Name</label>
                                    <input type="text" class="form-control" id="app_name" name="app_name" value="<?php echo htmlspecialchars($settings['app_name']); ?>" required>
                                </div>
                                
                                <div class="col-md-6">
                                    <label for="footer_text" class="form-label">Footer Text</label>
                                    <input type="text" class="form-control" id="footer_text" name="footer_text" value="<?php echo htmlspecialchars($settings['footer_text']); ?>">
                                </div>
                            </div>
                            
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="default_pagination" class="form-label">Default Pagination</label>
                                    <select class="form-select" id="default_pagination" name="default_pagination">
                                        <option value="5" <?php echo ($settings['default_pagination'] == 5) ? 'selected' : ''; ?>>5 items per page</option>
                                        <option value="10" <?php echo ($settings['default_pagination'] == 10) ? 'selected' : ''; ?>>10 items per page</option>
                                        <option value="20" <?php echo ($settings['default_pagination'] == 20) ? 'selected' : ''; ?>>20 items per page</option>
                                        <option value="50" <?php echo ($settings['default_pagination'] == 50) ? 'selected' : ''; ?>>50 items per page</option>
                                        <option value="100" <?php echo ($settings['default_pagination'] == 100) ? 'selected' : ''; ?>>100 items per page</option>
                                    </select>
                                </div>
                                
                                <div class="col-md-6">
                                    <label class="form-label d-block">System Logs</label>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="enable_logs" name="enable_logs" <?php echo ($settings['enable_logs'] == '1') ? 'checked' : ''; ?>>
                                        <label class="form-check-label" for="enable_logs">
                                            Enable System Logs
                                        </label>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-4">
                                <label for="logo" class="form-label">System Logo</label>
                                <input type="file" class="form-control" id="logo" name="logo" accept="image/*">
                                <div class="mt-2">
                                    <small class="text-muted">Current logo:</small>
                                    <div>
                                        <img src="<?php echo htmlspecialchars($settings['logo_path'] ?? 'assets/nsa-logo.jpeg'); ?>" alt="Current Logo" class="logo-preview" onerror="this.src='assets/nsa-logo.jpeg'">
                                    </div>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-1"></i> Save Settings
                            </button>
                        </form>
                    </div>
                    
                    <!-- System Info Tab -->
                    <div class="tab-pane fade" id="system-info" role="tabpanel" aria-labelledby="system-info-tab">
                        <div class="system-info">
                            <h5>System Information</h5>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="fw-bold">PHP Version:</label>
                                        <div><?php echo htmlspecialchars($php_version); ?></div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="fw-bold">MySQL Version:</label>
                                        <div><?php echo htmlspecialchars($mysql_version); ?></div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="fw-bold">Server Software:</label>
                                        <div><?php echo htmlspecialchars($_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'); ?></div>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="fw-bold">Application Version:</label>
                                        <div><?php echo htmlspecialchars($settings['app_version']); ?></div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="fw-bold">Database Name:</label>
                                        <div><?php echo htmlspecialchars($config['db_name']); ?></div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="fw-bold">Database Size:</label>
                                        <div><?php echo number_format($db_size / (1024 * 1024), 2); ?> MB</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <h5 class="mt-4">Content Statistics</h5>
                        <div class="row mt-3">
                            <div class="col-md-4">
                                <div class="card mb-3">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="me-3 fs-2 text-primary">
                                                <i class="fas fa-file-alt"></i>
                                            </div>
                                            <div>
                                                <div class="fs-4 fw-bold"><?php echo number_format($stats['reports'] ?? 0); ?></div>
                                                <div class="text-muted">Total Reports</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="card mb-3">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="me-3 fs-2 text-success">
                                                <i class="fas fa-chart-line"></i>
                                            </div>
                                            <div>
                                                <div class="fs-4 fw-bold"><?php echo number_format($stats['indicators'] ?? 0); ?></div>
                                                <div class="text-muted">Total Indicators</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="card mb-3">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="me-3 fs-2 text-info">
                                                <i class="fas fa-users"></i>
                                            </div>
                                            <div>
                                                <div class="fs-4 fw-bold"><?php echo number_format($stats['users'] ?? 0); ?></div>
                                                <div class="text-muted">Total Users</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Database Tab -->
                    <div class="tab-pane fade" id="database" role="tabpanel" aria-labelledby="database-tab">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i>
                            This section provides information about your database tables and allows you to perform maintenance tasks.
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="content-card mb-4">
                                    <div class="card-header">
                                        <h5 class="mb-0">Table Statistics</h5>
                                    </div>
                                    <div class="card-body p-0">
                                        <ul class="list-group list-group-flush">
                                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                                Reports
                                                <span class="badge bg-primary rounded-pill"><?php echo number_format($stats['reports'] ?? 0); ?> rows</span>
                                            </li>
                                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                                Report Categories
                                                <span class="badge bg-primary rounded-pill"><?php echo number_format($stats['categories'] ?? 0); ?> rows</span>
                                            </li>
                                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                                National Indicators
                                                <span class="badge bg-primary rounded-pill"><?php echo number_format($stats['indicators'] ?? 0); ?> rows</span>
                                            </li>
                                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                                Indicator Themes
                                                <span class="badge bg-primary rounded-pill"><?php echo number_format($stats['themes'] ?? 0); ?> rows</span>
                                            </li>
                                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                                Users
                                                <span class="badge bg-primary rounded-pill"><?php echo number_format($stats['users'] ?? 0); ?> rows</span>
                                            </li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="content-card mb-4">
                                    <div class="card-header">
                                        <h5 class="mb-0">Database Maintenance</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="d-grid gap-2">
                                            <button type="button" class="btn btn-outline-primary" disabled>
                                                <i class="fas fa-broom me-1"></i> Optimize Tables
                                            </button>
                                            <button type="button" class="btn btn-outline-warning" disabled>
                                                <i class="fas fa-sync me-1"></i> Repair Tables
                                            </button>
                                            <button type="button" class="btn btn-outline-info" disabled>
                                                <i class="fas fa-check-circle me-1"></i> Validate Tables
                                            </button>
                                            <div class="text-muted small text-center mt-2">
                                                <i class="fas fa-info-circle me-1"></i>
                                                These features will be available in a future update.
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Backup & Restore Tab -->
                    <div class="tab-pane fade" id="backup" role="tabpanel" aria-labelledby="backup-tab">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i>
                            Regular backups are essential to prevent data loss. We recommend backing up your database at least once a week.
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-4">
                                <div class="content-card">
                                    <div class="card-header">
                                        <h5 class="mb-0">Database Backup</h5>
                                    </div>
                                    <div class="card-body">
                                        <p>Generate a backup of your entire database. This will create a SQL file containing all your data.</p>
                                        <form>
                                            <div class="mb-3">
                                                <div class="form-check">
                                                    <input class="form-check-input" type="checkbox" id="include_data" checked disabled>
                                                    <label class="form-check-label" for="include_data">
                                                        Include data
                                                    </label>
                                                </div>
                                                <div class="form-check">
                                                    <input class="form-check-input" type="checkbox" id="include_structure" checked disabled>
                                                    <label class="form-check-label" for="include_structure">
                                                        Include table structure
                                                    </label>
                                                </div>
                                            </div>
                                            <button type="button" class="btn btn-primary" disabled>
                                                <i class="fas fa-download me-1"></i> Create Backup
                                            </button>
                                            <div class="text-muted small mt-2">
                                                <i class="fas fa-info-circle me-1"></i>
                                                This feature will be available in a future update.
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6 mb-4">
                                <div class="content-card">
                                    <div class="card-header">
                                        <h5 class="mb-0">Restore from Backup</h5>
                                    </div>
                                    <div class="card-body">
                                        <p>Restore your database from a previous backup file. <strong class="text-danger">Warning: This will overwrite your current data.</strong></p>
                                        <form>
                                            <div class="mb-3">
                                                <label for="backup_file" class="form-label">Backup File (SQL)</label>
                                                <input type="file" class="form-control" id="backup_file" disabled>
                                            </div>
                                            <button type="button" class="btn btn-warning" disabled>
                                                <i class="fas fa-upload me-1"></i> Restore Backup
                                            </button>
                                            <div class="text-muted small mt-2">
                                                <i class="fas fa-info-circle me-1"></i>
                                                This feature will be available in a future update.
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Footer -->
    <div class="footer">
        <p><?php echo $settings['app_name']; ?> &copy; <?php echo date('Y'); ?> <?php echo htmlspecialchars($settings['footer_text']); ?></p>
    </div>
    
    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mobile sidebar toggle
        document.getElementById('sidebarToggle')?.addEventListener('click', function() {
            document.querySelector('.sidebar').classList.toggle('active');
            document.querySelector('.main-content').classList.toggle('sidebar-active');
            document.querySelector('.footer').classList.toggle('sidebar-active');
        });
        
        // Logo preview
        document.getElementById('logo')?.addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file && file.type.match('image.*')) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const preview = document.querySelector('.logo-preview');
                    preview.src = e.target.result;
                }
                reader.readAsDataURL(file);
            }
        });
    </script>
</body>
</html>